from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector, \
    get_var_assignment_matches_containing_url_alias, contains_alias
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.evidences import FailedEvidence, LinkEvidence, Evidence, ComponentEvidence
from codeable_detectors.pyparsing_patterns import round_braces_block
from codeable_detectors.python.pythonDetectors import detect_python_import, get_import_patterns_from_imports_evidences, \
    get_python_variable_assignments
from codeable_detectors.utils import get_required_keyword_arg


# find evidence for a component by it being called using the requests API
class PythonRequestsCall(AtLeastOneFileMatchesDetector):
    def __init__(self, inside_call_patterns=None):
        super().__init__()
        self.file_endings = ["py"]
        self.inside_call_patterns = inside_call_patterns

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        is_detected = False

        import_matches = detect_python_import(ctx, "requests")
        if not import_matches:
            return FailedEvidence("python import for requests failed")
        matches.extend(import_matches)

        for import_pattern in get_import_patterns_from_imports_evidences(import_matches):
            # match something like: requests.get()
            requests_call_matches = ctx.matches_pattern(import_pattern + round_braces_block)
            if self.inside_call_patterns is None:
                if requests_call_matches:
                    matches.extend(requests_call_matches)
                    is_detected = True
                else:
                    return FailedEvidence("cannot find python requests call")
            else:
                for requests_call_match in requests_call_matches:
                    text = requests_call_match.text[requests_call_match.text.find("("):]
                    inside_call_matches = DetectorContext(text).matches_patterns(self.inside_call_patterns)
                    if inside_call_matches:
                        matches.append(requests_call_match)
                        is_detected = True

            if is_detected:
                return Evidence(matches)
        inside_string = ""
        if self.inside_call_patterns:
            inside_string = " with inside call pattern '" + str(self.inside_call_patterns) + "'"
        return FailedEvidence("cannot find python requests call" + inside_string)


class PythonRequestsCallToExternalComponent(PythonRequestsCall):
    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence
        return ComponentEvidence(evidence.matches).set_properties(
            detector_component_types=["service", "externalComponent"],
            detector_link_types=["restfulHTTP", "synchronousConnector"],
            detector_technology_types=["python", "requests"], kwargs=kwargs)


class PythonRequestsLink(PythonRequestsCall):
    def detect_in_context(self, ctx, **kwargs):
        target = get_required_keyword_arg('target', kwargs)
        matches = []

        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence

        var_assignments_matching_aliases = get_var_assignment_matches_containing_url_alias(
            get_python_variable_assignments(ctx), target.aliases)

        for request_call_match in evidence.matches:
            parameters = request_call_match.text[request_call_match.text.find("(") + 1:request_call_match.text.find(")")]
            if contains_alias(parameters, target.aliases):
                matches.append(request_call_match)
            else:
                # maybe a variable in the HTTP post method contains the url to the alias
                for var, var_assignment_match in var_assignments_matching_aliases:
                    if var in parameters:
                        matches.append(request_call_match)
                        matches.append(var_assignment_match)

        if matches:
            return LinkEvidence(matches).set_properties(
                # we add here sync for now, TODO: need to add python options to invokes requests in async fashion                                
                detector_link_types=["restfulHTTP", "synchronousConnector"],
                detector_technology_types=["python", "requests"], kwargs=kwargs)
        return FailedEvidence("no python request invocation to target found")
